'use strict';

/* --------------------------------------------------------------
 slider_responsive.js 2024-06-14
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2024 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Slider Responsive Module
 *
 * This module will handle the image replacement whenever the viewport breakpoint changes.
 */
gambio.widgets.module('slider_responsive', [gambio.source + '/libs/responsive', gambio.source + '/libs/events'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        defaults = {},
        options = $.extend(true, {}, defaults, data),
        module = {},
        slider;

    // ########## PRIVATE FUNCTIONS ##########

    var _breakpointHandler = function _breakpointHandler(event, currentBreakpoint) {
        var $swiperContainer = $this.find('.swiper-container'),
            $swiperWrapper = $swiperContainer.find('.swiper-wrapper'),
            previousSwiperInstance = $swiperContainer.get(0).swiper;

        // Reset the existing swiper instance (if any).  
        if (previousSwiperInstance) {
            try {
                previousSwiperInstance.destroy(true, true);
            } catch (exception) {
                // Sometime the breakpoint handler is called many times from various events which leads
                // to errors while destroying previous Swiper instances, thus the try-catch block. 
            }
        }
        $swiperWrapper.empty();
        $this.find('.swiper-pagination').empty();

        // Update the slider HTML markup with the breakpoint-respective image.
        slider.slides.forEach(function (slide) {
            slide.images.forEach(function (image) {
                if ((image.breakpoint === currentBreakpoint.name || image.breakpoint === 'xs' && currentBreakpoint.name === 'too small') && image.languageId === parseInt(jse.core.registry.get('languageId')) && image.image !== '') {

                    var $swiperSlide = $('<div class="swiper-slide"></div>');

                    // Are there image areas?
                    var hasAreas = image.areas && image.areas.length;

                    // Randomly generated string.
                    var imageMapId = Math.random().toString(36).substr(2, 5);

                    if (slide.thumbnail !== '') {
                        $swiperSlide.attr({
                            'data-thumb-image': jse.core.config.get('appUrl') + '/images/slider_images/thumbnails/' + slide.thumbnail,
                            'data-thumb-text': slide.title
                        });
                    }

                    var $slideImage = $('<img />');

                    // Use image map resizer plugin to adjust image map area sizes.
                    $slideImage.rwdImageMaps();

                    // Assign image map, if there are image areas.
                    if (hasAreas) {
                        $slideImage.attr('usemap', '#' + imageMapId);
                    }

                    $slideImage.attr({
                        class: 'img-responsive center-block',
                        src: jse.core.config.get('appUrl') + '/images/slider_images/' + image.image,
                        alt: slide.altText,
                        title: slide.title,
                        loading: 'lazy'
                    }).appendTo($swiperSlide);

                    if (slide.url) {
                        $slideImage.wrap('<a />').parent().attr({
                            href: slide.url,
                            target: slide.urlTarget
                        });
                    }

                    // Check for image areas and iterate over them.
                    if (hasAreas) {
                        // Create image map element.
                        var $map = $('<map name="' + imageMapId + '">');

                        /**
                         * Iterator function which processes every image area data.
                         * @param {Object} area Image area data.
                         */
                        var imageAreaIterator = function imageAreaIterator(area) {
                            var areaElementOptions = {
                                shape: 'poly',
                                coords: area.coordinates,
                                href: area.linkUrl,
                                title: area.linkTitle,
                                target: area.linkTarget,
                                'data-id': area.id
                            };

                            // Create image area element.
                            var $area = $('<area>', areaElementOptions);

                            // Put area into image map element.
                            $map.append($area);
                        };

                        // Process every image area.
                        image.areas.forEach(imageAreaIterator);

                        // Append image map to slide element.
                        $swiperSlide.append($map);
                    }

                    $swiperSlide.appendTo($swiperWrapper);
                }
            });
        });

        if ($swiperWrapper.children().length === 0) {
            return; // There is no slide set for this breakpoint. 
        }

        $swiperContainer.attr({
            'data-gambio-widget': 'swiper',
            'data-swiper-disable-translucence-fix': 'true'
        });

        $swiperContainer.data('swiper-breakpoints', [{
            breakpoint: 100,
            usePreviewBullets: true,
            slidesPerView: 1
        }]);

        $swiperContainer.data('swiper-slider-options', {
            effect: options.effect,
            speed: options.speed,
            nextButton: '.js-teaser-slider-next',
            prevButton: '.js-teaser-slider-prev',
            autoplay: slider.speed
        });

        // Initialize the new swiper instance and trigger the widget ready event. 
        gambio.widgets.init($swiperContainer);
        $('body').trigger(jse.libs.theme.events.SLIDER_RESPONSIVE_READY());

        $('img[usemap]').rwdImageMaps();
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget.
     */
    module.init = function (done) {
        if ($(options.source).length === 0) {
            return; // There is no JSON source for the slider data. 
        }

        slider = JSON.parse($(options.source).text());

        $(document).on('JSENGINE_INIT_FINISHED', function () {
            $('body').on(jse.libs.theme.events.BREAKPOINT(), _breakpointHandler);
            _breakpointHandler({}, jse.libs.theme.responsive.breakpoint());
            $('img[usemap]').rwdImageMaps();
        });

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
